var async               = require("async"),
    fs                  = require("fs-extra"),
    exec                = require("child_process").exec,
    systemctl           = require("systemctl-cmd"),
    mysql               = require("mysql2"),
    generator           = require("generate-password"),
    apt                 = require("apt"),
    config              = require("app/core/config")();

/*****************************************************************************\
    Return a set of functions which we can use to install and
    control components
\*****************************************************************************/
module.exports = function() {

    var _install_lamp = function(lamp_config, callback) {
        console.log("= Installing LAMP stack =");
        async.waterfall([
            function reload_config(next_install_lamp_step) {
                console.log(`Reloading config...`);
                config.load(function(error, data) {
                    if (!error) {
                        config = data;
                        console.log(`Reloaded config successfully`);
                    }
                    next_install_lamp_step(error);
                });
            },
            function check_mysql_installed(next_install_lamp_step) {
                if (!config.mysql) config.mysql = {'installed':false};
                exec(`sudo mysql --version`, function(error, stdout, stderr) {
                    if (!error) {
                        config.mysql.installed = true;
                        console.log("mysql is installed");
                    } else {
                        console.log("mysql is not installed");
                    }
                    next_install_lamp_step(null);
                });
            },
            function check_mysql_rootpass(next_install_lamp_step) {
                if (!lamp_config.db.rootpass && !(config.mysql && config.mysql.rootpass) ) {
                    console.log(`DB root pass is not available`);
                    lamp_config.db.rootpass = generator.generate({
                        length: 30,
                        numbers: true
                    });
                    console.log(`Generated new DB root pass: ${lamp_config.db.rootpass}`);
                } else if (config.mysql && config.mysql.rootpass) {
                    lamp_config.db.rootpass = config.mysql.rootpass;
                    console.log(`DB root pass found in config`);
                } else if (lamp_config.db.rootpass) {
                    console.log(`DB root pass found in lamp_config`);
                }
                next_install_lamp_step(null);
            },
            function install_mysql(next_install_lamp_step) {
                if (!config.mysql || !config.mysql.installed) {
                    console.log(`No installed mysql found`);
                    console.log(`Start mysql installation...`);
                    _install_mysql(lamp_config.db.rootpass, next_install_lamp_step)
                } else {
                    next_install_lamp_step(null);
                }
            },
            function reset_mysql_rootpass (next_install_lamp_step) {
                if (config.mysql && config.mysql.installed && !config.mysql.rootpass) {
                    console.log(`Unknown root pass of already installed mysql`);
                    console.log(`Start resetting mysql root pass...`);
                    _reset_mysql_rootpass(lamp_config.db.rootpass, next_install_lamp_step)
                } else {
                    next_install_lamp_step(null);
                }
            },
            function install_apache2(next_install_lamp_step) {
                _install_apache2(next_install_lamp_step);
            },
            function install_php(next_install_lamp_step) {
                _install_php(next_install_lamp_step);
            }
        ], callback);
    };


    var _install_php = function(callback) {
        console.log("= Installing PHP =");
        async.series([
            function install_package(next_install_php_step) {
                console.log("Installing php packages...");
                apt.install('php php-mbstring', function(error) {
                    if (!error) console.log("Packages installed successfully...");
                    next_install_php_step(error);
                });
            },
            function install_php_mysql(next_install_php_step) {
                if (config.mysql.installed) {
                    console.log(`mysql is installed`);
                    console.log(`Installing php-mysql...`);
                    apt.install('php-mysql', function(error) {
                        if (!error) console.log("Package installed successfully...");
                        next_install_php_step(error);
                    });
                } else {
                    next_install_php_step(null);
                }
            }
        ], function(error) {
            if (error) return callback(error);
            console.log(`Installed php successfully`);
            console.log(`Saving php info to config...`);
            config.load(function(error,config) {
                if (error) return callback(error);
                if (!config.php) config.php = {};
                config.php.installed = true;
                config.save(config, function(error) {
                if (!error) console.log(`Saved php info to config successfully`);
                    callback(error);
                });
            });
        });
    };

    var _is_php_installed = function(callback) {
        fs.pathExists('/usr/bin/php', callback);
    };



    var _install_apache2 = function(callback) {
        console.log("= Installing Apache2 =");
        async.series([
            function install_package(next_install_apache2_step) {
                console.log("Installing apache2 package...");
                apt.install('apache2', function(error) {
                    if (!error) console.log("Package installed successfully...");
                    next_install_apache2_step(error);
                });
            },
            function add_pi_to_www_group(next_install_akk_step) {
                console.log("Adding user pi to www-data group...");
                exec(`sudo adduser pi www-data`, function(error, stdout, stderr) {
                    if (!error) console.log("Added user pi to www-data group successfully");
                    next_install_akk_step(error);
                });
            },
            function recreate_www_dir(next_install_akk_step) {
                console.log("Recreating www dir if not exist...");
                exec(`sudo mkdir -p /var/www/`, function(error, stdout, stderr) {
                    if (!error) console.log("Recreated www dir successfully");
                    next_install_akk_step(error);
                });
            },
            function allow_write_perm_for_www_group(next_install_akk_step) {
                console.log("Allowing write permission to www-data group...");
                exec(`sudo chmod 775 /var/www/`, function(error, stdout, stderr) {
                    if (!error) console.log("Allowed write permission to www-data group successfully");
                    next_install_akk_step(error);
                });
            },
            function remove_default_html(next_install_apache2_step) {
                console.log("Removing apache2 default html...");
                exec(`sudo rm -rf /var/www/html/`, function(error, stdout, stderr) {
                    if (!error) console.log(`Removed apache2 default html successfully`);
                    next_install_apache2_step(error);
                })
            }
        ], function(error) {
            if (error) return callback(error);
            console.log(`Installed apache2 successfully`);
            console.log(`Saving apache2 info to config...`);
            config.load(function(error,config) {
                if (error) return callback(error);
                if (!config.apache2) config.apache2 = {};
                config.apache2.installed = true;
                config.save(config, function(error) {
                if (!error) console.log(`Saved apache2 info to config successfully`);
                    callback(error);
                });
            });
        });
    };

    var _enable_apache2 = function(callback) {
        console.log('Enable service...');
        systemctl.enable('apache2.service',true).then(result => {
            if (!result.error) console.log("Service enabled successfully...");
            callback(result.error);
        })
    };

    var _disable_apache2 = function(callback) {
        console.log('Disable service...');
        systemctl.disable('apache2.service',true).then(result => {
            if (!result.error) console.log("Service disabled successfully...");
            callback(result.error);
        })
    };

    var _start_apache2 = function(callback) {
        console.log('Start service...');
        systemctl.start('apache2.service',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service started successfully...");
                callback();
            }
        })
    };

    var _stop_apache2 = function(callback) {
        console.log('Stop service...');
        systemctl.stop('apache2.service',true).then(result => {
            if (!result.error) console.log("Service stopped successfully...");
            callback(result.error);
        })
    };

    var _restart_apache2 = function(callback) {
        console.log('Restarting service...');
        systemctl.restart('apache2.service',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service restarted successfully...");
                callback();
            }
        })
    };

    var _is_apache2_installed = function(callback) {
        fs.pathExists('/lib/systemd/system/apache2.service', callback);
    };

    var _is_apache2_enabled = function(callback) {
        systemctl.isEnabled('apache2.service').then(enabled => {
            callback(null, enabled);
        }, error => {
            callback(error);
        })
    };

    var _is_apache2_active = function(callback) {
        systemctl.isActive('apache2.service').then(active => {
            callback(null, active);
        }, error => {
            callback(error);
        })
    };



    var _install_mysql = function(rootpass, callback) {
        console.log("= Installing MySQL =");
        async.series([
            function set_debconf_selection(next_install_mysql_step) {
                exec(`sudo sh -c 'echo "mariadb-server mysql-server/root_password password ${rootpass}" | debconf-set-selections'`, function(error, stdout, stderr) {
                    if (!error) console.log(`Set debconf selection successfully...`);
                    next_install_mysql_step(error);
                })
            },
            function set_debconf_selection_again(next_install_mysql_step) {
                exec(`sudo sh -c 'echo "mariadb-server mysql-server/root_password_again password ${rootpass}" | debconf-set-selections'`, function(error, stdout, stderr) {
                    if (!error) console.log(`Set debconf selection successfully...`);
                    next_install_mysql_step(error);
                })
            },
            function install_package(next_install_mysql_step) {
                apt.install('mariadb-server', function(error) {
                    if (!error) console.log("Package installed successfully...");
                    next_install_mysql_step(error);
                });
            },
            function mysql_secure_installation(next_install_mysql_step) {
                exec(`sudo sh -c 'echo -e "${rootpass}\nn\ny\ny\ny\ny\n" | mysql_secure_installation'`, function(error, stdout, stderr) {
                    if (!error) console.log(`Processed mysql_secure_installation successfully...`);
                    next_install_mysql_step(error);
                })
            },
            function reset_mysql_rootpass (next_install_mysql_step) {
                _reset_mysql_rootpass(rootpass, next_install_mysql_step);
            }
        ], function(error) {
            if (error) return callback(error);
            console.log(`Installed mysql successfully`);
            console.log(`Saving mysql info to config...`);
            config.load(function(error,config) {
                if (error) return callback(error);
                if (!config.mysql) config.mysql = {};
                config.mysql.installed = true;
                config.mysql.rootpass = rootpass;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved mysql info to config successfully`);
                    callback(error);
                });
            });
        });
    };

    var _enable_mysql = function(callback) {
        console.log('Enable service...');
        systemctl.enable('mariadb.service',true).then(result => {
            if (!result.error) console.log("Service enabled successfully...");
            callback(result.error);
        })
    };

    var _disable_mysql = function(callback) {
        console.log('Disable service...');
        systemctl.disable('mariadb.service',true).then(result => {
            if (!result.error) console.log("Service disabled successfully...");
            callback(result.error);
        })
    };

    var _start_mysql = function(callback) {
        console.log('Start service...');
        systemctl.start('mariadb.service',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service started successfully...");
                callback();
            }
        })
    };

    var _stop_mysql = function(callback) {
        console.log('Stop service...');
        systemctl.stop('mariadb.service',true).then(result => {
            if (!result.error) console.log("Service stopped successfully...");
            callback(result.error);
        })
    };

    var _restart_mysql = function(callback) {
        console.log('Restarting service...');
        systemctl.restart('mariadb.service',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service restarted successfully...");
                callback();
            }
        })
    };

    var _reset_mysql_rootpass = function(rootpass, callback) {
        console.log("= Resetting mysql root pass =");
        var database_connection = null;
        async.series([
            function stop_mysql (next_reset_mysql_step) {
                console.log(`Stopping mysql service...`);
                systemctl.stop('mariadb',true).then(result => {
                    if (!result.error) console.log("Stopped mysql service successfully...");
                    next_reset_mysql_step(result.error);
                })
            },
            function start_mysql_safe (next_reset_mysql_step) {
                console.log(`Starting mysqld_safe...`);
                exec(`sudo mysqld_safe --skip-grant-tables --skip-networking`, function(error, stdout, stderr) {
//                        if (error) next_reset_mysql_step(error);
                })
                setTimeout(function(){ next_reset_mysql_step(); }, 10000);
            },
            function flush_database_privileges(next_reset_mysql_step) {
                console.log("Connecting to DB...");
                database_connection = mysql.createConnection({
                    host         : 'localhost',
                    user         : 'root',
                    socketPath   : '/var/run/mysqld/mysqld.sock'
                });
                console.log("Flushing privileges on DB...");
                var query = `FLUSH PRIVILEGES;`;
                database_connection.query(query, function (error, results, fields) {
                    if (!error) console.log(`Flushed privileges on DB successfully`);
                    next_reset_mysql_step(error);
                });
            },
            function change_database(next_reset_mysql_step) {
                console.log("Changing DB...");
                var query = `USE mysql;`;
                database_connection.query(query, function (error, results, fields) {
                    if (!error) console.log(`Changed DB successfully`);
                    next_reset_mysql_step(error);
                });
            },
            function changing_root_pass(next_reset_mysql_step) {
                console.log("Setting root pass on DB...");
                var query = `ALTER USER 'root'@'localhost' IDENTIFIED BY '${rootpass}';`;
                database_connection.query(query, function (error, results, fields) {
                    if (!error) console.log(`Set root pass on DB successfully`);
                    next_reset_mysql_step(error);
                });
            },
            function flush_database_privileges(next_reset_mysql_step) {
                console.log("Flushing privileges on DB...");
                var query = `FLUSH PRIVILEGES;`;
                database_connection.query(query, function (error, results, fields) {
                    if (!error) console.log(`Flushed privileges on DB successfully`);
                    database_connection.end();
                    next_reset_mysql_step(error);
                });
            },
            function kill_mysql_safe(next_reset_mysql_step) {
                console.log(`Stopping mysqld_safe...`);
                exec("sudo kill `sudo cat /var/run/mysqld/mysqld.pid`", function(error, stdout, stderr) {
                    if (!error) console.log(`Stopped mysqld_safe successfully`);
                    next_reset_mysql_step(error);
                })
                setTimeout(function(){ next_reset_mysql_step(); }, 10000);
            },
            function restart_mysql(next_reset_mysql_step) {
                console.log(`Restarting mysql...`);
                systemctl.restart('mariadb',true).then(result => {
                    if (!result.error) console.log("Restarted mysql service successfully...");
					setTimeout(function(){ next_reset_mysql_step(result.error); }, 5000);
                })
            }
        ], function(error) {
            if (error) return callback(error);
            console.log(`Resetted mysql root pass successfully`);
            console.log(`Saving mysql info to config...`);
            config.load(function(error,config) {
                if (error) return callback(error);
                if (!config.mysql) config.mysql = {};
                config.mysql.installed = true;
                config.mysql.rootpass = rootpass;
                config.save(config, function(error) {
                if (!error) console.log(`Saved mysql info to config successfully`);
                    callback(error);
                });
            });
        });
    };

    var _is_mysql_installed = function(callback) {
        fs.pathExists('/lib/systemd/system/mariadb.service', callback);
    };

    var _is_mysql_enabled = function(callback) {
        systemctl.isEnabled('mariadb.service').then(enabled => {
            callback(null, enabled);
        }, error => {
            callback(error);
        })
    };

    var _is_mysql_active = function(callback) {
        systemctl.isActive('mariadb.service').then(active => {
            callback(null, active);
        }, error => {
            callback(error);
        })
    };

    return {
        install_lamp:              _install_lamp,
        install_php:               _install_php,

        install_apache2:           _install_apache2,
        enable_apache2:            _enable_apache2,
        disable_apache2:           _disable_apache2,
        start_apache2:             _start_apache2,
        stop_apache2:              _stop_apache2,
        restart_apache2:           _restart_apache2,
        is_apache2_installed:      _is_apache2_installed,
        is_apache2_enabled:        _is_apache2_enabled,
        is_apache2_active:         _is_apache2_active,

        install_mysql:             _install_mysql,
        enable_mysql:              _enable_mysql,
        disable_mysql:             _disable_mysql,
        start_mysql:               _start_mysql,
        stop_mysql:                _stop_mysql,
        restart_mysql:             _restart_mysql,
        reset_mysql_rootpass:      _reset_mysql_rootpass,
        is_mysql_installed:        _is_mysql_installed,
        is_mysql_enabled:          _is_mysql_enabled,
        is_mysql_active:           _is_mysql_active,
    };
}
