var async           = require("async"),
    fs              = require("fs-extra"),
    exec            = require("child_process").exec,
    systemctl       = require("systemctl-cmd"),
    apt             = require("apt"),
    config          = require("app/core/config")(),
    helper          = require("app/core/helper");

/*****************************************************************************\
    Return a set of functions which we can use to install and
    control components
\*****************************************************************************/
module.exports = function() {

    var _install = function(companion_config, callback) {
        console.log("= Installing Companion =");
        if (!companion_config) var companion_config = {};
        console.log("Requested with config: ",JSON.stringify(companion_config,{},1));
        if (!companion_config.port) companion_config.port = 8000;
        console.log("Starts with config: ",JSON.stringify(companion_config,{},1));

        async.waterfall([
            function install_dependencies(next_step) {
                console.log('Installing dependencies...');
                apt.install('libgusb-dev curl git build-essential cmake libudev-dev libusb-1.0-0-dev', function(error) {
                    if (!error) console.log("Dependencies installed successfully...");
                    next_step(error);
                });
            },
            function reload_config(next_step) {
                console.log("Reloading config...");
                config.load(function(error, data) {
                    if (!error) {
                        config = data;
                        console.log("Reloaded config successfully");
                    }
                    next_step(error);
                });
            },
            function add_udev_rules(next_step) {
                console.log("Adding udev rules....");
                exec(`sudo cp -f ` + config.root + `/assets/companion/50-companion.rules /etc/udev/rules.d/50-companion.rules`, function(error) {
                    if (!error) console.log("Added udev rules successfully");
                    next_step(error);
                })
            },
            function reload_udev_rules(next_step) {
                console.log("Reloading the udev rules...");
                exec(`sudo udevadm control --reload-rules`, function(error, stdout, stderr) {
                    if (!error) console.log("Reloaded the udev rules successfully");
                    next_step(error);
                });
            },
            function install_yarn(next_step) {
                console.log("Installing yarn...");
                exec(`sudo npm install yarn -g`, function(error, stdout, stderr) {
                    if (!error) console.log("Installed yarn successfully");
                    next_step(error);
                });
            },
            function add_yarn_to_path(next_step) {
                console.log("Adding yarn to PATH...");
                exec(`export PATH="$HOME/.yarn/bin:$HOME/.config/yarn/global/node_modules/.bin:$PATH"`, function(error, stdout, stderr) {
                    if (!error) console.log("Added yarn to PATH successfully");
                    next_step(error);
                });
            },
            function cloning_git_repo(next_step) {
                console.log("Cloning git repo...");
                exec(`cd /home/pi && git clone https://github.com/bitfocus/companion`, function(error, stdout, stderr) {
                    if (!error) console.log("Cloned git repo successfully");
                    next_step(error);
                });
            },
            function build_companion(next_step) {
                console.log("Building Companion...");
                exec(`cd /home/pi/companion && yarn update && ./tools/build_writefile.sh`, function(error, stdout, stderr) {
                    if (!error) console.log("Builded companion successfully");
                    next_step(error);
                })
            },
            function create_service(next_step) {
                console.log("Creating service....");
                helper.writeTemplate(
                    config.root + "/assets/systemd/PiratenBox-Companion.service.template",
                    "/etc/systemd/system/PiratenBox-Companion.service",
                    companion_config, 
                    function(error) {
						if (!error) console.log("Service created successfully");
                        next_step(error);
                    }
                );
            },
            function fix_permissions(next_step) {
                console.log("Fixing permissions...");
                exec(`sudo chmod a+x /home/pi/companion/headless_ip.js`, function(error, stdout, stderr) {
                    if (!error) console.log("Fixed permissions successfully");
                    next_step(error);
                })
            },
            function reload_services(next_step) {
                console.log('Reload services...');
                systemctl.daemonReload(true).then(function(result) {
                    if (!result.error) console.log("Services reloaded successfully...");
                    next_step(result.error);
                });
            },
            function create_desktop_link(next_step) {
                console.log("Creating kiosk desktop link...");
                fs.pathExists(`/home/pi/Desktop/`, function(error, exists) {
                    if (exists) {
                        fs.copy(config.root + `/assets/desktop/PiratenBox-Companion.desktop`, `/home/pi/Desktop/PiratenBox-Companion.desktop`, next_step);
                    } else {
                        next_step(error);
                    }
                });
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
                if (!config.modules) config.modules = {};
                config.modules.companion = true;
                if (!config.companion) config.companion = {};
                config.companion.installed = true;
				config.companion.port = companion_config.port;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _enable = function(callback) {
        console.log('Enable service...');
        systemctl.enable('PiratenBox-Companion',true).then(result => {
            if (!result.error) console.log("Service enabled successfully...");
            callback(result.error);
        })
    };

    var _disable = function(callback) {
        console.log('Disable service...');
        systemctl.disable('PiratenBox-Companion',true).then(result => {
            if (!result.error) console.log("Service disabled successfully...");
            callback(result.error);
        })
    };

    var _start = function(callback) {
        console.log('Start service...');
        systemctl.start('PiratenBox-Companion',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service started successfully...");
                callback();
            }
        })
    };

    var _stop = function(callback) {
        console.log('Stop service...');
        systemctl.stop('PiratenBox-Companion',true).then(result => {
            if (!result.error) console.log("Service stopped successfully...");
            callback(result.error);
        })
    };

    var _restart = function(callback) {
        console.log('Restarting service...');
        systemctl.restart('PiratenBox-Companion',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service restarted successfully...");
                callback();
            }
        })
    };

    var _change_port = function(port, callback) {
        console.log("= Changing Companion port =");
        async.waterfall([
            function recreate_service(next_step) {
				console.log('Recreating service...');
				helper.writeTemplate(
					config.root + "/assets/systemd/PiratenBox-Companion.service.template",
					"/etc/systemd/system/PiratenBox-Companion.service",
					{port: port}, 
					function(error) {
						if (!error) console.log("Service recreated successfully");
						next_step(error);
					}
				);
            },
            function reload_services(next_step) {
                console.log('Reload services...');
                systemctl.daemonReload(true).then(function(result) {
                    if (!result.error) console.log("Services reloaded successfully...");
                    next_step(result.error);
                });
            },
            function restart_service(next_step) {
				console.log('Restarting service...');
				_restart(function(error) {
					if (!error) console.log("Service restarted successfully");
					next_step(error);
				});
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
				config.companion.port = port;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _get_port = function(callback) {
		if (config.companion && config.companion.port)
			callback(null, config.companion.port);
		else callback('Port not set in config');
    };

    var _is_installed = function(callback) {
        fs.pathExists('/etc/systemd/system/PiratenBox-Companion.service', callback);
    };

    var _is_enabled = function(callback) {
        systemctl.isEnabled('PiratenBox-Companion').then(enabled => {
            callback(null, enabled);
//            console.log((enabled ? 'Enabled' : 'Not enabled'));
        }, error => {
            callback(error);
        })
    };

    var _is_active = function(callback) {
        systemctl.isActive('PiratenBox-Companion').then(active => {
            callback(null, active);
//            console.log((active ? 'Active' : 'Not active'));
        }, error => {
            callback(error);
        })
    };

    return {
        install:                    _install,
        enable:                     _enable,
        disable:                    _disable,
        start:                      _start,
        stop:                       _stop,
        restart:                    _restart,
        get_port:                   _get_port,
        change_port:                _change_port,
        is_installed:               _is_installed,
        is_enabled:                 _is_enabled,
        is_active:                  _is_active,
    };
}