var async           = require("async"),
    fs              = require("fs-extra"),
    exec            = require("child_process").exec,
    systemctl       = require("systemctl-cmd"),
    apt             = require("apt"),
    config          = require("app/core/config")();

/*****************************************************************************\
    Return a set of functions which we can use to install and
    control components
\*****************************************************************************/
module.exports = function() {

    var _install = function(dicaffeine_config, callback) {
        console.log("= Installing dicaffeine =");
        if (!dicaffeine_config) var dicaffeine_config = {};
        console.log("Requested with config: ",JSON.stringify(dicaffeine_config,{},1));
        if (!dicaffeine_config.port) dicaffeine_config.port = 8080;
        console.log("Starts with config: ",JSON.stringify(dicaffeine_config,{},1));

        async.waterfall([
            function reload_config(next_step) {
                console.log("Reloading config...");
                config.load(function(error, data) {
                    if (!error) {
                        config = data;
                        console.log("Reloaded config successfully");
                    }
                    next_step(error);
                });
            },
            function install_repokey(next_step) {
                console.log("Installing apt repo key...");
                exec(`wget -O - http://dicaffeine.com/repository/dicaffeine.key | sudo apt-key add -`, function(error, stdout, stderr) {
                    if (!error) console.log("Installed repo key successfully");
                    next_step(error);
                });
            },
            function add_apt_repo(next_step) {
                console.log("Adding apt repo...");
                exec(`echo "deb https://dicaffeine.com/repository/ buster main non-free" | sudo tee -a /etc/apt/sources.list.d/dicaffeine.list`, function(error, stdout, stderr) {
                    if (!error) console.log("Added apt repo successfully");
                    next_step(error);
                });
            },
            function update_apt(next_step) {
                console.log("Updating apt...");
                apt.update(function(error) {
                    if (!error) console.log("Updated apt successfully");
                    next_step(error);
                });
            },
            function install_package(next_step) {
                console.log("Installing package...");
                apt.install(`dicaffeine`, function(error) {
                    if (!error) console.log("Installed package successfully");
                    next_step(error);
                })
            },
            function change_port(next_step) {
                console.log('Change port...');
				var file_name = "/etc/dicaffeine/dserver.json";
				var file_content = fs.readFileSync(file_name);
				var content = JSON.parse(file_content);
				content.port = dicaffeine_config.port;
				fs.writeFile(file_name, JSON.stringify(content, null, 4), function writeJSON(error) {
					if (error) return next_step(error);
					console.log(JSON.stringify(content, null, 1));
					console.log('written to ' + file_name);
					next_step();
				});
            },
            function restart_service(next_step) {
				console.log('Restarting service...');
				_restart(function(error) {
					if (!error) console.log("Service restarted successfully");
					next_step(error);
				});
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
                if (!config.modules) config.modules = {};
                config.modules.dicaffeine = true;
                if (!config.dicaffeine) config.dicaffeine = {};
                config.dicaffeine.installed = true;
				config.dicaffeine.port = dicaffeine_config.port;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _enable = function(callback) {
        console.log('Enable service...');
		exec(`XDG_RUNTIME_DIR=/run/user/1000 systemctl enable --user dicaffeine.service`, function(error, stdout, stderr) {
            if (!error) console.log("Service enabled successfully...");
			callback(error);
        })
    };

    var _disable = function(callback) {
        console.log('Disable service...');
		exec(`XDG_RUNTIME_DIR=/run/user/1000 systemctl disable --user dicaffeine.service`, function(error, stdout, stderr) {
            if (!error) console.log("Service disabled successfully...");
			callback(error);
        })
    };

    var _start = function(callback) {
        console.log('Start service...');
		exec(`XDG_RUNTIME_DIR=/run/user/1000 systemctl start --user dicaffeine.service`, function(error, stdout, stderr) {
            if (!error) console.log("Service started successfully...");
			callback(error);
        })
    };

    var _stop = function(callback) {
        console.log('Stop service...');
		exec(`XDG_RUNTIME_DIR=/run/user/1000 systemctl stop --user dicaffeine.service`, function(error, stdout, stderr) {
            if (!error) console.log("Service stopped successfully...");
			callback(error);
        })
    };

    var _restart = function(callback) {
        console.log('Restarting service...');
		exec(`XDG_RUNTIME_DIR=/run/user/1000 systemctl restart --user dicaffeine.service`, function(error, stdout, stderr) {
            if (!error) console.log("Service restarted successfully...");
			callback(error);
        })
    };

    var _change_port = function(port, callback) {
        console.log("= Changing Dicaffeine port =");
        async.waterfall([
            function change_port(next_step) {
                console.log('Change port...');
				var file_name = "/etc/dicaffeine/dserver.json";
				var file_content = fs.readFileSync(file_name);
				var content = JSON.parse(file_content);
				content.port = port;
				fs.writeFile(file_name, JSON.stringify(content, null, 4), function writeJSON(error) {
					if (error) return next_step(error);
					console.log(JSON.stringify(content, null, 1));
					console.log('written to ' + file_name);
					next_step();
				});
            },
            function restart_service(next_step) {
				console.log('Restarting service...');
				_restart(function(error) {
					if (!error) console.log("Service restarted successfully");
					next_step(error);
				});
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
				config.dicaffeine.port = port;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _get_port = function(callback) {
		if (config.dicaffeine && config.dicaffeine.port)
			callback(null, config.dicaffeine.port);
		else callback('Port not set in config');
    };

    var _is_installed = function(callback) {
        fs.pathExists('/home/pi/.config/systemd/user/default.target.wants/dicaffeine.service', callback);
    };

    var _is_enabled = function(callback) {
		exec(`XDG_RUNTIME_DIR=/run/user/1000 systemctl is-enabled --user dicaffeine.service`, function(error, stdout, stderr) {
			callback(error,(stdout.indexOf('enabled')===0));
        })
    };

    var _is_active = function(callback) {
		exec(`XDG_RUNTIME_DIR=/run/user/1000 systemctl is-active --user dicaffeine.service`, function(error, stdout, stderr) {
			callback(error,(stdout.indexOf('active')===0));
        })
    };

    return {
        install:                    _install,
        enable:                     _enable,
        disable:                    _disable,
        start:                      _start,
        stop:                       _stop,
        restart:                    _restart,
        get_port:                   _get_port,
        change_port:                _change_port,
        is_installed:               _is_installed,
        is_enabled:                 _is_enabled,
        is_active:                  _is_active,
    };
}
