var async           = require("async"),
    fs              = require("fs-extra"),
    exec            = require("child_process").exec,
    systemctl       = require("systemctl-cmd"),
    chrome          = require("chrome-launcher"),
    CDP             = require('chrome-remote-interface'),
    apt             = require("apt"),
    config          = require("app/core/config")();

/*****************************************************************************\
    Return the config and a function which we can use to change config file
\*****************************************************************************/
module.exports = function() {

    var is_LXDE = false;

    var _install = function(callback) {
        async.waterfall([
            function check_lxde(next_step) {
                console.log('Checking for LXDE...');
                fs.pathExists("/usr/share/xsessions/LXDE.desktop", function(error, exists) {
                    if (exists) is_LXDE = true;
                    next_step(error);
                });
            },
            function install_packages(next_step) {
                console.log('Installing kiosk packages...');
                if (is_LXDE) {
                    apt.install('unclutter nitrogen chromium-browser', function(error) {
                        if (!error) console.log("Packages installed successfully...");
                        next_step(error);
                    });
                } else {
                    apt.install('unclutter xorg xserver-xorg-legacy xserver-xorg-video-fbturbo pix-plym-splash openbox ratpoison nitrogen chromium-browser', function(error) {
                        if (!error) console.log("Packages installed successfully...");
                        next_step(error);
                    });
                }
            },
            function configure_displaymanager(next_step) {
                if (!is_LXDE) {
                    console.log('Configure display management...');
                    async.waterfall([
                        function remove_xauthority(next_display_step) {
                            exec(`sudo rm -f /home/pi/.XAuthority`, function(error) {
                                next_display_step(error);
                            });
                        },
                        function configure_xwrapper(next_display_step) {
                            exec(`sudo cp -f ` + config.root + `/assets/kiosk/Xwrapper.config /etc/X11/Xwrapper.config`, function(error) {
                                next_display_step(error);
                            });
                        },
/*                        function reconfigure_xorg(next_display_step) {
                            exec(`sudo dpkg-reconfigure xserver-xorg-legacy`, function(error) {
                                next_display_step(error);
                            });
                        },
*/                        function set_permissions(next_display_step) {
                            exec(`sudo usermod -aG video,tty,audio pi`, function(error) {
                                next_display_step(error);
                            });
                        },
                        function configure_ratpoison(next_display_step) {
                            fs.copy(config.root + `/assets/kiosk/.ratpoisonrc`, `/home/pi/.ratpoisonrc`, function(error) {
                                next_display_step(error);
                            });
                        },
                    ], function(error) {
                        if (error) console.log("Failed to configure display management");
                        if (!error) console.log("Configured display management successfully...");
                        next_step(error);
                    });
                } else {
                    next_step(null);
                }
            },
            function create_service(next_step) {
                console.log('Create service...');
                if (is_LXDE) {
                    exec(`sudo cp -f ` + config.root + `/assets/systemd/PiratenBox-Kiosk-LXDE.service /etc/systemd/system/PiratenBox-Kiosk.service`, function(error) {
                        if (!error) console.log("Service created successfully...");
                        next_step(error);
                    });
                } else {
                    exec(`sudo cp -f ` + config.root + `/assets/systemd/PiratenBox-Kiosk-Console.service /etc/systemd/system/PiratenBox-Kiosk.service`, function(error) {
                        if (!error) console.log("Service created successfully...");
                        next_step(error);
                    });
                }
            },
            function reload_services(next_step) {
                console.log('Reload services...');
                systemctl.daemonReload(true).then(function(result) {
                    if (!result.error) console.log("Services reloaded successfully...");
                    next_step(result.error);
                });
            },
/*            function disable_login_prompt(next_step) {
                console.log('Disable login prompt...');
                exec(`sudo systemctl disable getty@tty1.service`, function(error, stdout, stderr) {
                    if (!error) console.log("Disabled login prompt successfully...");
                    next_step(error);
                });
            },
*/            function create_desktop_link(next_step) {
                console.log('Creating kiosk desktop link...');
                fs.pathExists("/home/pi/Desktop/", function(error, exists) {
                    if (exists) {
                        fs.copy(config.root + `/assets/desktop/PiratenLiveBox-Kiosk.desktop`, `/home/pi/Desktop/PiratenLiveBox-Kiosk.desktop`, next_step);
                    } else {
                        next_step(error);
                    }
                });
            },
            function save_config(next_step) {
                console.log('Saving config...');
                if (!config.modules) config.modules = {};
                config.modules.kiosk = true;
                if (!config.kiosk) config.kiosk = {};
                config.kiosk.installed = true;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    callback(error);
                });
            },
        ], callback);
    };

    var _enable = function(callback) {
        console.log('Enable service...');
        systemctl.enable('PiratenBox-Kiosk',true).then(result => {
            if (!result.error) console.log("Service enabled successfully...");
            callback(result.error);
        })
    };

    var _disable = function(callback) {
        console.log('Disable service...');
        systemctl.disable('PiratenBox-Kiosk',true).then(result => {
            if (!result.error) console.log("Service disabled successfully...");
            callback(result.error);
        })
    };

    var _start = function(callback) {
        console.log('Start service...');
        systemctl.start('PiratenBox-Kiosk',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service started successfully...");
                callback();
            }
        })
    };

    var _stop = function(callback) {
        console.log('Stop service...');
        systemctl.stop('PiratenBox-Kiosk',true).then(result => {
            if (!result.error) console.log("Service stopped successfully...");
            callback(result.error);
        })
    };

    var _restart = function(callback) {
        console.log('Restarting service...');
        systemctl.restart('PiratenBox-Kiosk',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service restarted successfully...");
                callback();
            }
        })
    };

    var _is_installed = function(callback) {
        fs.pathExists('/etc/systemd/system/PiratenBox-Kiosk.service', callback);
    };

    var _is_enabled = function(callback) {
        systemctl.isEnabled('PiratenBox-Kiosk').then(enabled => {
            callback(null, enabled);
//            console.log((enabled ? 'Enabled' : 'Not enabled'));
        }, error => {
            callback(error);
        })
    };

    var _is_active = function(callback) {
        systemctl.isActive('PiratenBox-Kiosk').then(active => {
            callback(null, active);
//            console.log((active ? 'Active' : 'Not active'));
        }, error => {
            callback(error);
        })
    };

    var _save_instances = function(instances, callback) {
        console.log('Save instances to config...');
        config.load(function(error,config) {
            if (error) return callback(error);
            if (!config.kiosk) config.kiosk = {};
			config.kiosk.instances = instances;
			config.save(config, function(error) {
				if (!error) console.log(`Saved config successfully`);
				callback(error);
			});
        });
    };

    var _get_instances = function(callback) {
        console.log('Get instances from config...');
        config.load(function(error,config) {
            if (error) return callback(error);
            if (!config.kiosk) config.kiosk = {};
            if (config.kiosk&&config.kiosk.instances) callback(null, config.kiosk.instances)
            else callback(null, {});
        });
    };

    var _change_url = async function(instance, callback) {
        console.log('Change URL to "' + instance.url + '" on port ' + instance.port + '...');
        let client;
        try {
            // connect to endpoint
            client = await CDP({port:instance.port});
            // extract domains
            const {Network, Page} = client;
            // enable events then start!
            await Network.enable();
            await Page.enable();
            await Page.navigate({url: instance.url});
            await Page.loadEventFired();
            console.log('Loaded URL');
            callback();
        } catch (error) {
            console.error(error);
            callback(error);
        } finally {
            if (client) {
                await client.close();
                console.log('Closed CDP Client');
            }
        }
    };

    var _get_url = function(callback) {
        console.log('Get current URL from config...');
        config.load(function(error,config) {
            if (error) return callback(error);
            if (!config.kiosk) config.kiosk = {};
            if (config.kiosk&&config.kiosk.url) callback(null, config.kiosk.url)
            else callback(null, 'http://localhost:' + config.server.port + '/');
        });
    };

    var _request_url = function(callback) {
        console.log('Get current URL from chrome...');
        CDP.List((error, targets) => {
            if (!error) {
                for (let i = 0; i < targets.length; i++) {
                    if ( targets[i].type == 'page' ) {
                        callback(null, targets[i].url);
                        return;
                    }
                }
            }
            callback(error);
        });
    };

    return {
        install:                    _install,
        enable:                     _enable,
        disable:                    _disable,
        start:                      _start,
        stop:                       _stop,
        restart:                    _restart,
        is_installed:               _is_installed,
        is_enabled:                 _is_enabled,
        is_active:                  _is_active,
        change_url:                 _change_url,
        get_instances:              _get_instances,
        get_url:                    _get_url,
    };
}
