var async           = require("async"),
    fs              = require("fs-extra"),
    exec            = require("child_process").exec,
    systemctl       = require("systemctl-cmd"),
    apt             = require("apt"),
    config          = require("app/core/config")(),
    helper          = require("app/core/helper");

/*****************************************************************************\
    Return a set of functions which we can use to install and
    control components
\*****************************************************************************/
module.exports = function() {

    var _install = function(openslides_config, callback) {
        console.log("= Installing openslides =");
        if (!openslides_config) var openslides_config = {};
        console.log("Requested with config: ",JSON.stringify(openslides_config,{},1));
        if (!openslides_config.port) openslides_config.port = 9000;
        console.log("Starts with config: ",JSON.stringify(openslides_config,{},1));

        async.waterfall([
            function reload_config(next_step) {
                console.log(`Reloading config...`);
                config.load(function(error, data) {
                    if (!error) {
                        config = data;
                        console.log(`Reloaded config successfully`);
                    }
                    next_step(error);
                });
            },
            function install_requirements(next_step) {
                apt.install('python3-pip python3-dev build-essential libffi-dev libssl-dev', function(error) {
                    if (!error) console.log("Packages installed successfully...");
                    next_step(error);
                });
            },
            function install_openslides(next_step) {
                console.log("Installing openslides package...");
                exec(`pip3 install openslides`, function(error, stdout, stderr) {
                    if (!error) console.log("Installed openslides package successfully");
                    next_step(error);
                });
            },
            function create_service(next_step) {
                console.log('Create service...');
                helper.writeTemplate(
                    config.root + "/assets/systemd/PiratenBox-OpenSlides.service.template",
                    "/etc/systemd/system/PiratenBox-OpenSlides.service",
                    openslides_config, 
                    function(error) {
						if (!error) console.log("Service created successfully");
                        next_step(error);
                    }
                );
            },
            function reload_services(next_step) {
                console.log('Reload services...');
                systemctl.daemonReload(true).then(function(result) {
                    if (!result.error) console.log("Services reloaded successfully...");
                    next_step(result.error);
                });
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
	            config.load(function(error,config) {
	                if (error) return callback(error);
	                if (!config.modules) config.modules = {};
	                config.modules.openslides = true;
	                if (!config.openslides) config.openslides = {};
	                config.openslides.installed = true;
					config.openslides.port = openslides_config.port;
	                config.save(config, function(error) {
	                    if (!error) console.log(`Saved config successfully`);
	                    next_step(error);
	                });
	            });
            }
        ], callback);
    };

    var _enable = function(callback) {
        console.log('Enable service...');
        systemctl.enable('PiratenBox-OpenSlides',true).then(result => {
            if (!result.error) console.log("Service enabled successfully...");
            callback(result.error);
        })
    };

    var _disable = function(callback) {
        console.log('Disable service...');
        systemctl.disable('PiratenBox-OpenSlides',true).then(result => {
            if (!result.error) console.log("Service disabled successfully...");
            callback(result.error);
        })
    };

    var _start = function(callback) {
        console.log('Start service...');
        systemctl.start('PiratenBox-OpenSlides',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service started successfully...");
                callback();
            }
        })
    };

    var _stop = function(callback) {
        console.log('Stop service...');
        systemctl.stop('PiratenBox-OpenSlides',true).then(result => {
            if (!result.error) console.log("Service stopped successfully...");
            callback(result.error);
        })
    };

    var _restart = function(callback) {
        console.log('Restarting service...');
        systemctl.restart('PiratenBox-OpenSlides',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service restarted successfully...");
                callback();
            }
        })
    };

    var _change_port = function(port, callback) {
        console.log("= Changing OpenSlides port =");
        async.waterfall([
            function stop_service(next_step) {
				console.log('Stopping service...');
				_stop(function(error) {
					if (!error) console.log("Service stopped successfully");
					next_step(error);
				});
            },
            function recreate_service(next_step) {
				console.log('Recreating service...');
				helper.writeTemplate(
					config.root + "/assets/systemd/PiratenBox-OpenSlides.service.template",
					"/etc/systemd/system/PiratenBox-OpenSlides.service",
					{port: port}, 
					function(error) {
						if (!error) console.log("Service recreated successfully");
						next_step(error);
					}
				);
            },
            function reload_services(next_step) {
                console.log('Reload services...');
                systemctl.daemonReload(true).then(function(result) {
                    if (!result.error) console.log("Services reloaded successfully...");
                    next_step(result.error);
                });
            },
            function start_service(next_step) {
				console.log('Restarting service...');
				_start(function(error) {
					if (!error) console.log("Service started successfully");
					next_step(error);
				});
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
				config.openslides.port = port;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _get_port = function(callback) {
		if (config.openslides && config.openslides.port)
			callback(null, config.openslides.port);
		else callback('Port not set in config');
    };

    var _is_installed = function(callback) {
        fs.pathExists('/etc/systemd/system/PiratenBox-OpenSlides.service', callback);
    };

    var _is_enabled = function(callback) {
        systemctl.isEnabled('PiratenBox-OpenSlides').then(enabled => {
            callback(null, enabled);
        }, error => {
            callback(error);
        })
    };

    var _is_active = function(callback) {
        systemctl.isActive('PiratenBox-OpenSlides').then(active => {
            callback(null, active);
        }, error => {
            callback(error);
        })
    };

    return {
        install:                    _install,
        enable:                     _enable,
        disable:                    _disable,
        start:                      _start,
        stop:                       _stop,
        restart:                    _restart,
        get_port:                   _get_port,
        change_port:                _change_port,
        is_installed:               _is_installed,
        is_enabled:                 _is_enabled,
        is_active:                  _is_active,
    };
}
