var async           = require("async"),
    fs              = require("fs-extra"),
    exec            = require("child_process").exec,
    systemctl       = require("systemctl-cmd"),
    apt             = require("apt"),
    config          = require("app/core/config")(),
    helper          = require("app/core/helper");

/*****************************************************************************\
    Return a set of functions which we can use to install and
    control components
\*****************************************************************************/
module.exports = function() {

    var _install = function(vnc_config, callback) {
        console.log("= Installing VNC =");
        if (!vnc_config) var vnc_config = {};
        console.log("Requested with config: ",JSON.stringify(vnc_config,{},1));
        if (!vnc_config.port) vnc_config.port = 5900;
        if (!vnc_config.passwd) vnc_config.passwd = 'PiratenBox';
        console.log("Starts with config: ",JSON.stringify(vnc_config,{},1));

        async.waterfall([
            function reload_config(next_step) {
                console.log("Reloading config...");
                config.load(function(error, data) {
                    if (!error) {
                        config = data;
                        console.log("Reloaded config successfully");
                    }
                    next_step(error);
                });
            },
			function install_package(next_step) {
                console.log('Installing package...');
                apt.install('x11vnc', function(error) {
                    if (!error) console.log("Package installed successfully...");
                    next_step(error);
                });
            },
            function store_passwd(next_step) {
                console.log("Storing VNC password...");
                exec(`sudo sh -c 'printf "${vnc_config.passwd}\n${vnc_config.passwd}\ny\n" | x11vnc -storepasswd /etc/x11vnc.pwd'`, function(error, stdout, stderr) {
                    if (!error) console.log("Stored VNC password successfully");
                    next_step(error);
                });
            },
            function cloning_websockify_git_repo(next_step) {
                console.log("Cloning websockify git repo...");
                exec(`cd /home/pi && git clone https://github.com/novnc/websockify`, function(error, stdout, stderr) {
                    if (!error) console.log("Cloned websockify git repo successfully");
                    next_step(error);
                });
            },
            function install_websockify(next_step) {
                console.log("Installing websockify...");
                exec(`cd /home/pi/websockify && sudo python3 setup.py install`, function(error, stdout, stderr) {
                    if (!error) console.log("Installed websockify successfully");
                    next_step(error);
                })
            },
            function create_service(next_step) {
                console.log("Creating service....");
                helper.writeTemplate(
                    config.root + "/assets/systemd/PiratenBox-VNC.service.template",
                    "/etc/systemd/system/PiratenBox-VNC.service",
                    vnc_config, 
                    function(error) {
						if (!error) console.log("Service created successfully");
                        next_step(error);
                    }
                );
            },
            function reload_services(next_step) {
                console.log('Reload services...');
                systemctl.daemonReload(true).then(function(result) {
                    if (!result.error) console.log("Services reloaded successfully...");
                    next_step(result.error);
                });
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
                if (!config.modules) config.modules = {};
                config.modules.vnc = true;
                if (!config.vnc) config.vnc = {};
                config.vnc.installed = true;
				config.vnc.port = vnc_config.port;
				config.vnc.passwd = vnc_config.passwd;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _enable = function(callback) {
        console.log('Enable service...');
        systemctl.enable('PiratenBox-VNC',true).then(result => {
            if (!result.error) console.log("Service enabled successfully...");
            callback(result.error);
        })
    };

    var _disable = function(callback) {
        console.log('Disable service...');
        systemctl.disable('PiratenBox-VNC',true).then(result => {
            if (!result.error) console.log("Service disabled successfully...");
            callback(result.error);
        })
    };

    var _start = function(callback) {
        console.log('Start service...');
        systemctl.start('PiratenBox-VNC',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service started successfully...");
                callback();
            }
        })
    };

    var _stop = function(callback) {
        console.log('Stop service...');
        systemctl.stop('PiratenBox-VNC',true).then(result => {
            if (!result.error) console.log("Service stopped successfully...");
            callback(result.error);
        })
    };

    var _restart = function(callback) {
        console.log('Restarting service...');
        systemctl.restart('PiratenBox-VNC',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("Service restarted successfully...");
                callback();
            }
        })
    };

    var _set_port = function(port, callback) {
        console.log("= Set VNC port =");
        async.waterfall([
            function recreate_service(next_step) {
				console.log('Recreating service...');
				helper.writeTemplate(
					config.root + "/assets/systemd/PiratenBox-VNC.service.template",
					"/etc/systemd/system/PiratenBox-VNC.service",
					{port: port}, 
					function(error) {
						if (!error) console.log("Service recreated successfully");
						next_step(error);
					}
				);
            },
            function reload_services(next_step) {
                console.log('Reload services...');
                systemctl.daemonReload(true).then(function(result) {
                    if (!result.error) console.log("Services reloaded successfully...");
                    next_step(result.error);
                });
            },
            function restart_service(next_step) {
				console.log('Restarting service...');
				_restart(function(error) {
					if (!error) console.log("Service restarted successfully");
					next_step(error);
				});
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
				config.vnc.port = port;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _get_port = function(callback) {
		if (config.vnc && config.vnc.port)
			callback(null, config.vnc.port);
		else callback('Port not set in config');
    };

    var _set_passwd = function(passwd, callback) {
        console.log("= Set VNC password =");
        async.waterfall([
            function store_passwd(next_step) {
                console.log("Storing VNC password...");
                exec(`sudo sh -c 'echo -e "${passwd}\n${passwd}\ny\n" | x11vnc -storepasswd /etc/x11vnc.pwd'`, function(error, stdout, stderr) {
                    if (!error) console.log("Stored VNC password successfully");
                    next_step(error);
                });
            },
            function restart_service(next_step) {
				console.log('Restarting service...');
				_restart(function(error) {
					if (!error) console.log("Service restarted successfully");
					next_step(error);
				});
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
				config.vnc.passwd = passwd;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _get_passwd = function(callback) {
		if (config.vnc && config.vnc.passwd)
			callback(null, config.vnc.passwd);
		else callback('Password not set in config');
    };

    var _is_installed = function(callback) {
        fs.pathExists('/etc/systemd/system/PiratenBox-VNC.service', callback);
    };

    var _is_enabled = function(callback) {
        systemctl.isEnabled('PiratenBox-VNC').then(enabled => {
            callback(null, enabled);
//            console.log((enabled ? 'Enabled' : 'Not enabled'));
        }, error => {
            callback(error);
        })
    };

    var _is_active = function(callback) {
        systemctl.isActive('PiratenBox-VNC').then(active => {
            callback(null, active);
//            console.log((active ? 'Active' : 'Not active'));
        }, error => {
            callback(error);
        })
    };

    return {
        install:                    _install,
        enable:                     _enable,
        disable:                    _disable,
        start:                      _start,
        stop:                       _stop,
        restart:                    _restart,
        set_port:                   _set_port,
        get_port:                   _get_port,
        set_passwd:                 _set_passwd,
        get_passwd:                 _get_passwd,
        is_installed:               _is_installed,
        is_enabled:                 _is_enabled,
        is_active:                  _is_active,
    };
}