var async               = require("async"),
    exec                = require("child_process").exec,
    alsa                = require("node-alsa-ctl"),
    apt                 = require("apt"),
    ps                  = require("ps-node");


/*****************************************************************************\
    Return a set of functions which we can use to manage audio
\*****************************************************************************/
module.exports = function() {

    var amixer = `XDG_RUNTIME_DIR=/run/user/${process.getuid()} amixer`;
    _isNumeric = function(n) {
        return !isNaN(parseFloat(n)) && isFinite(n);
    }

    var _get_devices = function(callback) {
        alsa.getDevices()
            .then( (devices) => {
                callback(null, devices);
            })
            .catch( (err) => {
                callback(err);
            });
    }

    var _get_soundcards = function(callback) {
        alsa.getSoundcards()
            .then( (soundcards) => {
                callback(null, soundcards);
            })
            .catch( (err) => {
                callback(err);
            });
    }

    var _get_controls = function(soundcard, callback) {
        alsa.getControls(soundcard)
            .then( (controls) => {
                callback(null, controls);
            })
            .catch( (err) => {
                callback(err);
            });
    }
    var _get_control = function(soundcard, numid, callback) {
        alsa.getControls(soundcard, numid)
            .then( (controls) => {
                callback(null, controls);
            })
            .catch( (err) => {
                callback(err);
            });
    }
    var _set_control = function(soundcard, numid, values, callback) {
        alsa.setControl(soundcard,{'numid':numid,'values':values})
            .then( (controls) => {
                callback(null, controls);
            })
            .catch( (err) => {
                callback(err);
            });
    }

    var _parse_amixer_channel = function(inorout, line) {
        regex = new RegExp(inorout + " ([-\\d]+) \\[(\\d+)%] \\[([-.\\d]+)dB] \\[(on|off)]")
        match = line.match(regex)
        if (match) {
            return {
                volume: Number(match[1]),
                level: Number(match[2]),
                decibel: Number(match[3]),
                active: (match[4]=='on')
            }
        }
        regex = new RegExp(inorout + " ([-\\d]+) \\[(\\d+)%] \\[(on|off)]")
        match = line.match(regex)
        if (match) {
            return {
                volume: Number(match[1]),
                level: Number(match[2]),
                active: (match[3]=='on')
            }
        }
        regex = new RegExp(inorout + " \\[(on|off)]")
        match = line.match(regex)
        if (match) {
            return {
                active: (match[1]=='on')
            }
        }
    }

    var _parse_simple_controls = function(scontents) {
        const data = [];
        let currentControl = null;
        scontents.split('\n').forEach(line => {
            const matchControl = line.match(/^([^']+) '([^']+)',(\d+)$/);
            if (matchControl) {
                if (currentControl) {
                    data.push(currentControl);
                }
                let [, type, name, index] = matchControl;
                index = Number(index);
                currentControl = {type, name, index};
                return;
            }
            const matchValue = line.match(/^ {2}([^:]+): ?(.*)$/);
            if (matchValue) {
                const [, key, val] = matchValue;
                currentControl[key] = val;
            }
        });
        data.push(currentControl);
        data.map(m => {
            if (!m) return;

            let match = m.name.match(/(\d{2})\. (.*)/);
            if (match) {
                const [, num, name] = match;
                m.shortname = name;
                m.num = Number(num);
            }

            if (m['Capabilities']) {
                m.capabilities = m['Capabilities'].split(' ');
            }
            delete m['Capabilities'];

            if (m['Playback channels']) {
                m.playback = {};
            }
            delete m['Playback channels'];

            if (m['Capture channels']) {
                m.capture = {};
            }
            delete m['Capture channels'];

            if (m['Front Left']) {
                if (m.playback) m.playback['Front Left'] = _parse_amixer_channel('Playback',m['Front Left']);
                if (m.capture) m.capture['Front Left'] = _parse_amixer_channel('Capture',m['Front Left']);
            }
            delete m['Front Left'];

            if (m['Front Right']) {
                if (m.playback) m.playback['Front Right'] = _parse_amixer_channel('Playback',m['Front Right']);
                if (m.capture) m.capture['Front Right'] = _parse_amixer_channel('Capture',m['Front Right']);
            }
            delete m['Front Right'];

            if (m['Mono']) {
                if (m.playback) m.playback['Mono'] = _parse_amixer_channel('Playback',m['Mono']);
                if (m.capture) m.capture['Mono'] = _parse_amixer_channel('Capture',m['Mono']);
            }
            delete m['Mono'];

            if (m['Limits']) {
                match = m['Limits'].match(/Playback ([-\d]+) - ([-\d]+)/);
                if (match) {
                    var limits = {
                        min: Number(match[1]),
                        max: Number(match[2])
                    }
                    if (m.playback['Front Left']) m.playback['Front Left'].limits = limits;
                    if (m.playback['Front Right']) m.playback['Front Right'].limits = limits;
                    if (m.playback['Mono']) m.playback['Mono'].limits = limits;
                }
                match = m['Limits'].match(/Capture ([-\d]+) - ([-\d]+)/);
                if (match) {
                    var limits = {
                        min: Number(match[1]),
                        max: Number(match[2])
                    }
                    if (m.capture['Front Left']) m.capture['Front Left'].limits = limits;
                    if (m.capture['Front Right']) m.capture['Front Right'].limits = limits;
                    if (m.capture['Mono']) m.capture['Mono'].limits = limits;
                }
            }
            delete m['Limits'];

            return m;
        });
        return data;
    }
    var _get_simple_controls = function(soundcard, callback) {
        var command = amixer;
        if ( _isNumeric(soundcard) ) command += ` -c ${soundcard}`;
        exec(command + ` scontents`, function(error, stdout, stderr) {
            if (error) callback(error);
            else callback(null, _parse_simple_controls(stdout));
        });
    }
    var _get_simple_control = function(soundcard, name, callback) {
        var command = amixer;
        if ( _isNumeric(soundcard) ) command += ` -c ${soundcard}`;
        exec(command + ` sget "${name}"`, function(error, stdout, stderr) {
            if (error) callback(error);
            else callback(null, _parse_simple_controls(stdout));
        });
    }
    var _set_simple_control = function(soundcard, name, values, callback) {
        var command = amixer;
        if ( _isNumeric(soundcard) ) command += ` -c ${soundcard}`;
        exec(command + ` sset "${name}" ${values.join(' ')}`, function(error, stdout, stderr) {
            if (error) callback(error);
            else callback(null, _parse_simple_controls(stdout));
        });
    }

    var _is_pulseaudio = function(callback) {
        ps.lookup({ command: 'pulseaudio' }, function(error, result ) {
            if (error) return callback(error);
            var process = result[ 0 ];
            if (process) {
                console.log('audio - is_pulseaudio',!!process);
                callback(null, true);
            } else {
                console.log('audio - No pulseaudio process found');
                callback(null, false);
            }
        });
    }

    var _get_sink_list = function(callback) {
        _is_pulseaudio(function(error,is_pulseaudio) {
            if (!is_pulseaudio) return callback('No pulseaudio');
            let command = `XDG_RUNTIME_DIR=/run/user/${process.getuid()} pacmd list-sinks | grep -e index: -e name: -e alsa.name -e 'alsa\\.card = ' | sed s/*//g | sed s/^[' '\\\t]*//g | sed s/'index: '//g | sed s/'name: <'//g | sed s/'>'//g | sed s/'alsa.name = '//g | sed s/'alsa.card = '//g | sed s/\\"//g | tr '\\n' '/'`;
            exec(command, function(error, stdout, stderr) {
                if (error) return callback(error);
                var values = stdout.split('/');
                var sinklist = [];
                for (i=0;i<=(values.length-4);i=i+4) {
                    sinklist.push({
                        'index': values[i].trim(),
                        'name': values[i+1].trim(),
                        'cardName': values[i+2].trim(),
                        'cardNumber': values[i+3].trim()
                    });
                }
//                console.log('audio - sinklist',sinklist)
                callback(null, sinklist);
            });
        });
    }

    var _get_default_output = function(callback) {
        _is_pulseaudio(function(error,is_pulseaudio){
            if (is_pulseaudio) {
                let command = `cat /home/pi/.config/pulse/*-default-sink`;
                exec(command, function(error, stdout, stderr) {
                    if (error) return callback(error,stderr);
                    var default_sink = stdout.trim();
//                    console.log('audio - default_sink',default_sink);
                    _get_sink_list(function(error, sinklist) {
                        if (error) return callback(error);
                        var found = false;
                        sinklist.forEach(function(sink){
                            if (sink.name == default_sink) {
                                found = true;
//                                console.log('audio - default_output',sink.cardName);
                                return callback(null, sink);
                            }
                        });
                        if (!found) return callback('not found');
                    });
                });
            } else {
                return callback('is not pulse - not implemented yet');
            }
        });
    }

    var _set_default_output = function(soundcard, callback) {
        _is_pulseaudio(function(error,is_pulseaudio){
            if (is_pulseaudio) {
                _get_sink_list(function(error, sinklist) {
                    if (error) return callback(error);
                    var found = false;
                    console.log('audio - sinklist',sinklist);
                    sinklist.forEach(function(sink){
                        if (sink.cardNumber == soundcard) {
                            found = true;
                            let command = `XDG_RUNTIME_DIR=/run/user/${process.getuid()} pactl set-default-sink ${sink.index}`;
                            exec(command, function(error, stdout, stderr) {
                                if (error) return callback(error);
                                _get_default_output(function(error,default_output){
                                    console.log('audio - default_output',default_output);
                                    return callback(error, default_output);
                                });
                            });
                        }
                    });
                    if (!found) return callback('not found');
                });
            } else {
                return callback('is not pulse - not implemented yet');
            }
        });
    }

/*
aplay -l | grep -q "bcm2835 ALSA";
amixer cset numid=3 "$AUDIO_OUT"

/home/pi/.asoundrc
pcm.!default {
  type asym
  playback.pcm {
    type plug
    slave.pcm "output"
  }
  capture.pcm {
    type plug
    slave.pcm "input"
  }
}

pcm.output {
  type hw
  card $AUDIO_OUT
}

ctl.!default {
  type hw
  card $AUDIO_OUT
}


speaker-test -Dplughw:Headphones -c2 -t wav -l1
*/

    return {
        get_devices:                _get_devices,
        get_soundcards:             _get_soundcards,
        get_controls:               _get_controls,
        get_control:                _get_control,
        set_control:                _set_control,
        get_simple_controls:        _get_simple_controls,
        get_simple_control:         _get_simple_control,
        set_simple_control:         _set_simple_control,
        get_default_output:         _get_default_output,
        set_default_output:         _set_default_output,
    };

}
