var async               = require("async"),
    fs                  = require("fs-extra"),
    exec                = require("child_process").exec,
    systemctl           = require("systemctl-cmd"),
    bluetooth           = require("bluetoothctlwe");
bluetooth.Bluetooth()

/*****************************************************************************\
    Return a set of functions which we can use to manage and check bluetooth
    connection information
\*****************************************************************************/
module.exports = function() {
	let _pin = null;

    var _is_installed = function(callback) {
        fs.pathExists('/lib/systemd/system/bluetooth.service', callback);
    };

    var _block = function(callback) {
        console.log('bluetooth - Block radio...');
		exec(`rfkill block bluetooth`, function(error, stdout, stderr) {
			if (!error) {
				console.log("bluetooth - Radio blocked successfully");
				bluetooth.Bluetooth()
				callback( null, stdout );
			} else {
				console.log("bluetooth - Failed to block the radio");
				callback( error, stderr );
			}
		});
    };
    var _unblock = function(callback) {
        console.log('bluetooth - Unblock radio...');
		exec(`rfkill unblock bluetooth`, function(error, stdout, stderr) {
			if (!error) {
				console.log("bluetooth - Radio unblocked successfully");
				bluetooth.Bluetooth()
				callback( null, stdout );
			} else {
				console.log("bluetooth - Failed to unblock the radio");
				callback( error, stderr );
			}
		});
    };
    var _is_blocked = function(callback) {
        console.log('bluetooth - Get  block radio state...');
		exec(`rfkill --json`, function(error, stdout, stderr) {
			if ( !error && stdout ) {
				let all_devices = JSON.parse( stdout );
				let bluetooth_devices = all_devices[""].filter(function( obj ) {
					return obj.type == "bluetooth";
				});
				let blocked_state = ( bluetooth_devices[0].soft == 'blocked' || bluetooth_devices[0].hard == 'blocked' )
				callback( null, blocked_state );
			} else {
				console.log("bluetooth - Failed to get radio block state");
				callback( error, stderr );
			}
		});
    };

    var _enable = function(callback) {
        console.log('bluetooth - Enable service...');
        systemctl.enable('bluetooth.service',true).then(result => {
            if (!result.error) console.log("bluetooth - Service enabled successfully...");
            callback(result.error);
        })
    };
    var _disable = function(callback) {
        console.log('bluetooth - Disable service...');
        systemctl.disable('bluetooth.service',true).then(result => {
            if (!result.error) console.log("bluetooth - Service disabled successfully...");
            callback(result.error);
        })
    };
    var _is_enabled = function(callback) {
        systemctl.isEnabled('bluetooth.service').then(enabled => {
            callback(null, enabled);
        }, error => {
            callback(error);
        })
    };

    var _start = function(callback) {
        console.log('bluetooth - Start service...');
        systemctl.start('bluetooth.service',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("bluetooth - Service started successfully...");
                callback();
            }
        })
    };
    var _stop = function(callback) {
        console.log('bluetooth - Stop service...');
        systemctl.stop('bluetooth.service',true).then(result => {
            if (!result.error) console.log("bluetooth - Service stopped successfully...");
            callback(result.error);
        })
    };
    var _restart = function(callback) {
        console.log('bluetooth - Restarting service...');
        systemctl.restart('bluetooth.service',true).then(result => {
            if (result.error) callback(result.error)
            else {
                console.log("bluetooth - Service restarted successfully...");
                callback();
            }
        })
    };
    var _is_active = function(callback) {
        systemctl.isActive('bluetooth.service').then(active => {
            callback(null, active);
        }, error => {
            callback(error);
        })
    };

    var _is_available = function( callback ) {
		callback( null, bluetooth.checkBluetoothController() );
    };

    var _is_ready = function( callback ) {
		callback( null, bluetooth.isBluetoothReady );
    };

    var _is_discoverable = function( callback ) {
		callback( null, bluetooth.isDiscoverable );
    };
    var _set_discoverable = function( value, callback ) {
		callback( null, bluetooth.discoverable(value) );
    };

    var _is_powered = function( callback ) {
		callback( null, bluetooth.isPowered );
    };
    var _set_powered = function( value, callback ) {
		callback( null, bluetooth.power(value) );
    };
    var _reset = function( callback ) {
        async.series([
			function(next_step) {
				_set_powered( false, next_step );
			},
			function(next_step) {
				_set_powered( true, next_step );
			},
        ], callback);
    };

    var _is_pairable = function( callback ) {
		callback( null, bluetooth.isPairable );
    };
    var _set_pairable = function( value, callback ) {
		callback( null, bluetooth.pairable(value) );
    };

    var _is_scanning = function( callback ) {
		callback( null, bluetooth.isScanning );
    };
    var _set_scanning = function( value, callback ) {
		callback( null, bluetooth.scan(value) );
    };

    var _pair = function( mac, callback ) {
		callback( null, bluetooth.pair( mac ) );
    };
    var _unpair = function( mac, callback ) {
		callback( null, bluetooth.remove( mac ) );
    };
    var _is_confirming = function( callback ) {
		callback( null, bluetooth.isConfirmingPassKey );
    };
    var _is_waiting_for_pin = function( callback ) {
		callback( null, bluetooth.isWaitingForPIN );
    };
    var _get_pin = function( callback ) {
		callback( null, _pin );
    };

    var _connect = function( mac, callback ) {
		callback( null, bluetooth.connect( mac ) );
    };
    var _disconnect = function( mac, callback ) {
		callback( null, bluetooth.disconnect( mac ) );
    };

    var _get_controllers = function( callback ) {
		callback( null, bluetooth.controllers );
    };

    var _get_scanned_devices = function( callback ) {
		callback( null, bluetooth.devices );
    };

    var _get_paired_devices = function( callback ) {
		let paired_devices = bluetooth.devices.filter(function( obj ) {
			return obj.paired == "yes";
		});
		callback( null, paired_devices);
    };
    var _get_trusted_devices = function( callback ) {
		var trusted_devices = bluetooth.devices.filter(function( obj ) {
			return obj.trusted == "yes";
		});
		callback( null, trusted_devices);
    };
    var _get_blocked_devices = function( callback ) {
		callback( null, function() {
			return bluetooth.devices.filter(function( obj ) {
				return obj.blocked == "yes";
			});
		});
    };
    var _get_connected_devices = function( callback ) {
		callback( null, function() {
			return bluetooth.devices.filter(function( obj ) {
				return obj.connected == "yes";
			});
		});
    };


	bluetooth.on(bluetooth.bluetoothEvents.PassKey, function (passkey) {
		console.log('bluetooth - Confirm passkey:' + passkey)
		bluetooth.confirmPassKey(true);
	})

	bluetooth.on(bluetooth.bluetoothEvents.PIN, function (passkey) {
		console.log('bluetooth - Enter PIN:' + passkey)
		_pin = passkey;
	})

	bluetooth.on(bluetooth.bluetoothEvents.Paired, function (passkey) {
		console.log('bluetooth - Successfully paired')
		_pin = null;
	})


    return {
        is_installed:               _is_installed,

        block:                      _block,
        unblock:                    _unblock,
        is_blocked:                 _is_blocked,

        enable:                     _enable,
        disable:                    _disable,
        is_enabled:                 _is_enabled,

        start:                      _start,
        stop:                       _stop,
        restart:                    _restart,
        is_active:                  _is_active,

        is_available:               _is_available,
        is_ready:                   _is_ready,

        is_discoverable:            _is_discoverable,
        set_discoverable:           _set_discoverable,

        is_powered:                 _is_powered,
        set_powered:                _set_powered,
        reset:                      _reset,

        is_pairable:                _is_pairable,
        set_pairable:               _set_pairable,

        is_scanning:                _is_scanning,
        set_scanning:               _set_scanning,

        pair:                       _pair,
        unpair:                     _unpair,
        is_confirming:              _is_confirming,
        is_waiting_for_pin:         _is_waiting_for_pin,
        get_pin:                    _get_pin,

        connect:                    _connect,
        disconnect:                 _disconnect,

        get_controllers:            _get_controllers,
        get_scanned_devices:        _get_scanned_devices,
        get_paired_devices:         _get_paired_devices,

        get_trusted_devices:        _get_trusted_devices,
        get_blocked_devices:        _get_blocked_devices,
        get_connected_devices:      _get_connected_devices,
    };
}
