var async               = require("async"),
    exec                = require("child_process").exec,
    systemctl           = require("systemctl-cmd"),
    xrandrParser        = require("xrandr").parser;

/*****************************************************************************\
    Return a set of functions which we can use to install and
    control components
\*****************************************************************************/
module.exports = function() {

    var _set_on = function(callback) {
        console.log('system - Switch screen on...');
        async.waterfall([
            function monitor_on(next_step) {
                exec(`xset -display :0 dpms force on`, function(error, stdout, stderr) {
                    next_step(error);
                });
            },
            function reset_screensaver(next_step) {
                exec(`xset -display :0 s reset`, function(error, stdout, stderr) {
                    next_step(error);
                });
            },
            function unblank_screensaver(next_step) {
                exec(`xset -display :0 s noblank`, function(error, stdout, stderr) {
                    next_step(error);
                });
            },
            function back_to_kiosk_if_active(next_step) {
                systemctl.isActive('PiratenBox-Kiosk').then(enabled => {
                    console.log('system - Is Kiosk active: ' + enabled);
                    if (enabled) {
                        exec(`xset -display :0 s off`, function(error, stdout, stderr) {
                            if (error) return next_step(error);
                            exec(`xset -display :0 -dpms`, function(error, stdout, stderr) {
                                next_step(error);
                            });
                        });
                    } else {
                        next_step();
                    }
                }, error => {
                    next_step(error);
                })

            }
        ], callback);
    };

    var _set_off = function(callback) {
        console.log('system - Switch screen off...');
        async.waterfall([
            function monitor_off(next_step) {
                exec(`xset -display :0 dpms force off`, function(error, stdout, stderr) {
                    next_step(error);
                });
            },
            function blank_screensaver(next_step) {
                exec(`xset -display :0 s blank`, function(error, stdout, stderr) {
                    next_step(error);
                });
            },
            function reset_screensaver(next_step) {
                exec(`xset -display :0 s reset`, function(error, stdout, stderr) {
                    next_step(error);
                });
            },
            function activate_screensaver(next_step) {
                exec(`xset -display :0 s activate`, function(error, stdout, stderr) {
                    next_step(error);
                });
            },
        ], callback);
    };

    var _is_on = function(callback) {
//        console.log('system - Check if screen is on...');
        exec(`xset -display :0 -q | grep "Monitor is" | awk '{print $3}'`, function(error, stdout, stderr) {
            if (error) callback(error);
            exec(`xset -display :0 -q | grep "blanking" | awk '{print $3}'`, function(error2, stdout2, stderr2) {
                if (error) callback(error);
                var result = {
                    'monitor': stdout.toLowerCase().trim(),
                    'blanking': stdout2.toLowerCase().trim(),
                };
                callback(error, result);
            });
        });
    };


	var _get_monitors = function(callback) {
		exec('xrandr', {env: {'DISPLAY': ':0.0'}}, (error, stdout) => {
			if (error) {
				callback(error);
			} else {
				var monitor_info = xrandrParser(stdout);
				callback(null, monitor_info);
			}
		});
	};

	var _get_monitor = function(monitor, callback) {
		_get_monitors(function(error, result) {
			var monitor_info = result[monitor] || null;
//			console.log('system - monitor_info: ', monitor_info);
			callback(null, monitor_info);
		});
	};

	var _set_monitor = function(monitor, options, callback) {
		if(!monitor) return callback('No monitor supplied');
		var cmd = `xrandr --output ${monitor} ${options.join(' ')}`;
		console.log('system - cmd: ', cmd);
		exec(cmd, {env: {'DISPLAY': ':0.0'}}, (error, stdout, stderr) => {
			if (error) return callback(error);
			if (stderr) return callback(stderr);
			exec(`autorandr --save manual`, {env: {'DISPLAY': ':0.0'}}, (error, stdout, stderr) => {
				if (error) return callback(error);
				if (stderr) return callback(stderr);
				_get_monitor(monitor, callback);
			});
		});
	};

	var _set_monitor_on = function(monitor, callback) {
		_set_monitor(monitor, ['--auto'], callback);
	};

	var _set_monitor_off = function(monitor, callback) {
		_set_monitor(monitor, ['--off'], callback);
	};

	var _set_monitor_resolution = function(monitor, resolution, callback) {
		_set_monitor(monitor, ['--mode', resolution], callback);
	};

	var _set_monitor_rotation = function(monitor, rotation, callback) {
		_set_monitor(monitor, ['--rotate', rotation], callback);
	};

	var _set_monitor_position = function(monitor, position, callback) {
		_set_monitor(monitor, ['--pos', position], callback);
	};

	var _set_monitor_primary = function(monitor, callback) {
		_set_monitor(monitor, ['--primary'], callback);
	};


	var _get_blanking = function(callback) {
		exec(`raspi-config nonint get_blanking`, function(error, stdout, stderr) {
			callback(error, (stdout.trim()=='1'));
		});
	};

	var _set_blanking = function(blanking, callback) {
        async.waterfall([
            function do_blanking(next_step) {
				var blanking_char = (blanking)?'1':'0';
				var blanking_exec = "sudo raspi-config nonint do_blanking " + blanking_char;
				exec(blanking_exec, function(error, stdout, stderr) {
					next_step(error, (stdout.trim()=='1'));
				});
            },
            function reboot(result, next_step) {
				exec(`sudo reboot`, function(error, stdout, stderr) {
					callback(error, result);
				});
            },
        ], callback);
	};


	var _get_overscan = function(callback) {
		exec(`raspi-config nonint get_overscan`, function(error, stdout, stderr) {
			callback(error, (stdout.trim()=='1'));
		});
	};

	var _set_overscan = function(overscan, callback) {
        async.waterfall([
            function do_overscan(next_step) {
				var overscan_char = (overscan)?'1':'0';
				var overscan_exec = "sudo raspi-config nonint do_overscan " + overscan_char;
				exec(overscan_exec, function(error, stdout, stderr) {
					next_step(error, (stdout.trim()=='1'));
				});
            },
            function reboot(result, next_step) {
				exec(`sudo reboot`, function(error, stdout, stderr) {
					callback(error, result);
				});
            },
        ], callback);
	};


    return {
        is_on:                      _is_on,
        set_on:                     _set_on,
        set_off:                    _set_off,
        set_monitor_on:             _set_monitor_on,
        set_monitor_off:            _set_monitor_off,
        set_monitor_resolution:     _set_monitor_resolution,
        set_monitor_rotation:       _set_monitor_rotation,
        set_monitor_position:       _set_monitor_position,
        set_monitor_primary:        _set_monitor_primary,
        get_monitors:               _get_monitors,
        get_monitor:                _get_monitor,
        get_overscan:               _get_overscan,
        set_overscan:               _set_overscan,
    };
}
