var async           = require("async"),
    fs              = require("fs-extra"),
    exec            = require("child_process").exec,
    systemctl       = require("systemctl-cmd"),
    mysql           = require('mysql2'),
    generator       = require('generate-password'),
    config          = require("app/core/config")(),
    helper          = require("app/core/helper"),
    lamp            = require("app/core/lamp")();

/*****************************************************************************\
    Return a set of functions which we can use to install and
    control components
\*****************************************************************************/
module.exports = function() {

    var _install = function(akk_config, callback) {
        console.log("= Installing akk =");
        if (!akk_config) var akk_config = {};
        console.log("Requested with config: ",JSON.stringify(akk_config,{},1));

        if (!akk_config.db) akk_config.db = {};
        if (!akk_config.db.name) akk_config.db.name = "piratenbox_akk";
        if (!akk_config.db.user) akk_config.db.user = "piratenbox_akk";
        if (!akk_config.db.pass) {
            akk_config.db.pass = generator.generate({
            	length: 30,
            	numbers: true
            });
        }
        if (!akk_config.event) akk_config.event = {};
        if (!akk_config.event.name) akk_config.event.name = "";
        if (!akk_config.event.start) akk_config.event.start = "";
        if (!akk_config.event.end) akk_config.event.end = "";
        if (!akk_config.event.location) akk_config.event.location = "";
        if (!akk_config.event.level) akk_config.event.level = "";
        if (!akk_config.event.PT) akk_config.event.PT = 0;
        if (!akk_config.event.AV) akk_config.event.AV = 0;
        if (!akk_config.system) akk_config.system = {};
        if (!akk_config.system.rootdir) akk_config.system.rootdir = "/var/www";
        if (!akk_config.system.port) akk_config.system.port = "1337";

        console.log("Starts with config: ",JSON.stringify(akk_config,{},1));

        var database_connection = null;
        async.waterfall([
            function install_lamp(next_install_akk_step) {
                lamp.install_lamp(akk_config, next_install_akk_step);
            },
            function reload_config(next_install_akk_step) {
                console.log(`Reloading config...`);
                config.load(function(error, data) {
                    if (!error) {
                        config = data;
                        console.log(`Reloaded config successfully`);
                    }
                    next_install_akk_step(error);
                });
            },
            function remove_web_dir(next_install_akk_step) {
                console.log(`Removing web dir before cloning...`);
                exec(`sudo rm -rf ${akk_config.system.rootdir}/web/`, function(error, stdout, stderr) {
                    if (!error) console.log("Fixed ownership successfully");
                    next_install_akk_step(error);
                });
            },
            function clone_repo(next_install_akk_step) {
                console.log("Cloning akk repo for installation...");
                exec(`sudo git clone https://github.com/DJaeger/akk.git ${akk_config.system.rootdir}/web/`, function(error, stdout, stderr) {
                    if (!error) console.log("Cloned akk repo for installation successfully");
                    next_install_akk_step(error);
                });
            },
            function create_data_dir(next_install_akk_step) {
                console.log("Creating data dir...");
                exec(`sudo mkdir -p ${akk_config.system.rootdir}/data`, function(error, stdout, stderr) {
                    if (!error) console.log("Created data dir successfully");
                    next_install_akk_step(error);
                });
            },
            function create_upload_dir(next_install_akk_step) {
                console.log("Creating upload dir...");
                exec(`sudo mkdir -p ${akk_config.system.rootdir}/upload`, function(error, stdout, stderr) {
                    if (!error) console.log("Created upload dir successfully");
                    next_install_akk_step(error);
                });
            },
            function fix_permissions(next_install_akk_step) {
                console.log("Fixing ownership...");
                exec(`sudo chown -R www-data:www-data ${akk_config.system.rootdir}/`, function(error, stdout, stderr) {
                    if (!error) console.log("Fixed ownership successfully");
                    next_install_akk_step(error);
                });
            },
            function disable_default_ports(next_install_akk_step) {
                console.log("Disabling Apache default ports...");
                helper.writeTemplate(
                    config.root + "/assets/akk/ports.conf.template",
                    "/etc/apache2/ports.conf",
                    akk_config, 
                    function(error) {
                        if (!error) console.log("Disabled Apache default ports successfully");
                        next_install_akk_step(error);
                    }
                );
            },
            function configure_default_site(next_install_akk_step) {
                console.log("Configuring Apache default site...");
                helper.writeTemplate(
                    config.root + "/assets/akk/default.conf.template",
                    "/etc/apache2/sites-available/000-default.conf",
                    akk_config, 
                    function(error) {
                        if (!error) console.log("Configured Apache default site successfully");
                        next_install_akk_step(error);
                    }
                );
            },
            function drop_database(next_install_akk_step) {
                console.log("Connecting to DB...");
                database_connection = mysql.createConnection({
                    host         : 'localhost',
                    user         : 'root',
                    password     : config.mysql.rootpass,
                    socketPath   : '/var/run/mysqld/mysqld.sock'
                });
                console.log("Dropping DB if exists...");
                var query = `DROP DATABASE IF EXISTS ${akk_config.db.name};`;
                database_connection.query(query, function (error, results, fields) {
                    if (!error) console.log("Dropped existing akk DB successfully");
                    next_install_akk_step(error);
                });
            },
            function create_database(next_install_akk_step) {
                console.log("Creating DB...");
                var query = `CREATE DATABASE ${akk_config.db.name} CHARSET='utf8' COLLATE='utf8_general_ci';`;
                database_connection.query(query, function (error, results, fields) {
                    if (!error) console.log("Created akk DB successfully");
                    next_install_akk_step(error);
                });
            },
            function grant_database_privileges(next_install_akk_step) {
                console.log("Granting privileges on DB...");
                var query = `GRANT ALL ON ${akk_config.db.name}.* TO ${akk_config.db.user}@localhost IDENTIFIED BY '${akk_config.db.pass}';`;
                database_connection.query(query, function (error, results, fields) {
                    if (!error) console.log("Granted privileges on DB successfully");
                    next_install_akk_step(error);
                });
            },
            function flush_database_privileges(next_install_akk_step) {
                console.log("Flushing privileges on DB...");
                var query = `FLUSH PRIVILEGES;`;
                database_connection.query(query, function (error, results, fields) {
                    database_connection.end();
                    if (!error) console.log("Flushed privileges on DB successfully");
                    next_install_akk_step(error);
                });
            },
            function configure_akk(next_install_akk_step) {
                console.log("Saving DB info to akk config...");
                helper.writeTemplate(
                    config.root + "/assets/akk/akk.ini.template",
                    akk_config.system.rootdir + "/web/inc/akk.ini",
                    akk_config, 
                    function(error) {
                        if (!error) console.log("Saved DB info to akk config successfully");
                        next_install_akk_step(error);
                    }
                );
            },
            function restart_apache2(next_install_akk_step) {
                console.log(`Restarting apache2...`);
                systemctl.restart('apache2.service',true).then(result => {
                    if (result.error) callback(result.error)
                    else {
                        console.log("Restarted apache2 successfully");
                        next_install_akk_step(null);
                    }
                })
            },
            function save_config(next_install_akk_step) {
                console.log(`Saving config...`);
                if (!config.modules) config.modules = {};
                config.modules.akk = true;
                if (!config.akk) config.akk = {};
                config.akk.installed = true;
                config.akk.port = akk_config.system.port;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_install_akk_step(error);
                });
            }
        ], callback);
    };

    var _change_port = function(akk_config, callback) {
        console.log("= Changing Akk-Tool port =");
        if (!akk_config.system) akk_config.system = {};
        if (!akk_config.system.rootdir) akk_config.system.rootdir = config.akk.rootdir || "/var/www";
        if (!akk_config.system.port) akk_config.system.port = "1337";
        async.waterfall([
            function setting_default_ports(next_install_akk_step) {
                console.log("Setting Apache ports...");
                helper.writeTemplate(
                    config.root + "/assets/akk/ports.conf.template",
                    "/etc/apache2/ports.conf",
                    akk_config, 
                    function(error) {
                        if (!error) console.log("Disabled Apache default ports successfully");
                        next_install_akk_step(error);
                    }
                );
            },
            function reconfigure_default_site(next_install_akk_step) {
                console.log("Reconfiguring Apache default site...");
                helper.writeTemplate(
                    config.root + "/assets/akk/default.conf.template",
                    "/etc/apache2/sites-available/000-default.conf",
                    akk_config, 
                    function(error) {
                        if (!error) console.log("Reconfigured Apache default site successfully");
                        next_install_akk_step(error);
                    }
                );
            },
            function restart_apache2(next_install_akk_step) {
                console.log(`Restarting apache2...`);
                systemctl.restart('apache2.service',true).then(result => {
                    if (result.error) callback(result.error)
                    else {
                        console.log("Restarted apache2 successfully");
                        next_install_akk_step(null);
                    }
                })
            },
            function save_config(next_step) {
                console.log(`Saving config...`);
				config.akk.port = akk_config.system.port;
                config.save(config, function(error) {
                    if (!error) console.log(`Saved config successfully`);
                    next_step(error);
                });
            }
        ], callback);
    };

    var _get_port = function(callback) {
		if (config.akk && config.akk.port)
			callback(null, config.akk.port);
		else callback('Port not set in config');
    };

    var _is_installed = function(callback) {
        if (config.modules) {
            callback(null,!!config.modules.akk);
        } else {
            callback(null,false);
        }
    };

    return {
        is_installed:               _is_installed,
        change_port:                _change_port,
        get_port:                   _get_port,
        install:                    _install,
    };
}
